"""
Doom Launcher - Kivy GUI for Pygame Raycaster
"""

import multiprocessing as mp
import math
from game import game_process

# Kivy Imports
from kivy.app import App
from kivy.uix.boxlayout import BoxLayout
from kivy.uix.gridlayout import GridLayout
from kivy.uix.button import Button
from kivy.uix.label import Label
from kivy.uix.slider import Slider
from kivy.uix.switch import Switch
from kivy.uix.tabbedpanel import TabbedPanel, TabbedPanelItem
from kivy.uix.scrollview import ScrollView
from kivy.lang import Builder


# CONFIGURATION DEFAULTS

DEFAULT_CONFIG = {
    # Display
    'WIDTH': 800, 'HEIGHT': 600, 'FOV': 2 * math.pi / 3, 'NUM_RAYS': 200, 'MAX_DEPTH': 20,
    
    # Player
    'player_speed': 2.0, 'player_rotation_speed': 0.05, 'player_start_x': 3.5, 'player_start_y': 14.5,
    'player_start_angle': 0.0, 'player_health': 100,
    
    # Keybinds (WASD)
    'key_forward': 119, 'key_backward': 115, 'key_left': 97, 'key_right': 100,
    'key_rotate_left': 276, 'key_rotate_right': 275, 'key_shoot': 32,
    
    # Enemies
    'num_enemies': 6, 'enemy_size': 0.3,
    
    # Map (16x16)
    'map': [
        [1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1],
        [1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1],
        [1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1],
        [1, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1],
        [1, 0, 0, 0, 1, 1, 1, 0, 0, 0, 0, 0, 0, 0, 0, 1],
        [1, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1],
        [1, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1],
        [1, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1],
        [1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1],
        [1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 1],
        [1, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 1, 0, 0, 0, 1],
        [1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 1],
        [1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1],
        [1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1],
        [1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1],
        [1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1]
    ],

    # Graphics
    'show_minimap': True, 'show_hud': True,
}


# KIVY UI

class LauncherUI(BoxLayout):
    def __init__(self, shared_config, command_queue, game_proc_container, **kwargs):
        super().__init__(**kwargs)
        self.shared_config = shared_config
        self.command_queue = command_queue
        self.game_proc_container = game_proc_container

        # Single panel with all controls
        self.add_widget(self.create_main_panel())
    
    def create_main_panel(self):
        """Creates a single panel with all controls"""
        scroll = ScrollView()
        layout = BoxLayout(orientation='vertical', padding=20, spacing=15, size_hint_y=None)
        layout.bind(minimum_height=layout.setter('height'))

        # Title
        layout.add_widget(Label(text='Doom', font_size=24, bold=True, size_hint_y=None, height=40))

        # Status
        self.status_label = Label(text='Status: Not Running', font_size=16, size_hint_y=None, height=30)
        layout.add_widget(self.status_label)

        # Game Control Buttons
        btn_layout = BoxLayout(size_hint_y=None, height=50, spacing=10)
        start_btn = Button(text='Start Game', font_size=16, on_press=self.start_game)
        stop_btn = Button(text='Stop Game', font_size=16, on_press=self.stop_game)
        restart_btn = Button(text='Restart', font_size=16, on_press=self.restart_game)

        btn_layout.add_widget(start_btn)
        btn_layout.add_widget(stop_btn)
        btn_layout.add_widget(restart_btn)
        layout.add_widget(btn_layout)

        # Settings Section
        layout.add_widget(Label(text='Settings', font_size=18, bold=True, size_hint_y=None, height=30))

        # FOV Slider
        layout.add_widget(Label(text='FOV (degrees)', size_hint_y=None, height=25))
        slider_fov = Slider(min=30, max=120, value=math.degrees(self.shared_config['FOV']), step=5, size_hint_y=None, height=30)
        label_fov = Label(text=f"{math.degrees(self.shared_config['FOV']):.1f}", size_hint_y=None, height=25)
        slider_fov.bind(value=lambda s, v: self.update_config_and_label('FOV', v, label_fov))

        slider_box_fov = BoxLayout(orientation='horizontal', size_hint_y=None, height=30, spacing=10)
        slider_box_fov.add_widget(slider_fov)
        slider_box_fov.add_widget(label_fov)
        layout.add_widget(slider_box_fov)

        # Player Speed
        layout.add_widget(Label(text='Player Speed', size_hint_y=None, height=25))
        slider_speed = Slider(min=1.0, max=5.0, value=self.shared_config['player_speed'], step=0.5, size_hint_y=None, height=30)
        label_speed = Label(text=f"{self.shared_config['player_speed']:.1f}", size_hint_y=None, height=25)
        slider_speed.bind(value=lambda s, v: self.update_config_and_label('player_speed', v, label_speed))

        slider_box_speed = BoxLayout(orientation='horizontal', size_hint_y=None, height=30, spacing=10)
        slider_box_speed.add_widget(slider_speed)
        slider_box_speed.add_widget(label_speed)
        layout.add_widget(slider_box_speed)

        # Enemies
        layout.add_widget(Label(text='Enemies', size_hint_y=None, height=25))
        slider_enemies = Slider(min=0, max=12, value=self.shared_config['num_enemies'], step=1, size_hint_y=None, height=30)
        label_enemies = Label(text=str(self.shared_config['num_enemies']), size_hint_y=None, height=25)
        slider_enemies.bind(value=lambda s, v: self.update_config_and_label('num_enemies', int(v), label_enemies))

        slider_box_enemies = BoxLayout(orientation='horizontal', size_hint_y=None, height=30, spacing=10)
        slider_box_enemies.add_widget(slider_enemies)
        slider_box_enemies.add_widget(label_enemies)
        layout.add_widget(slider_box_enemies)

        # UI Toggles
        layout.add_widget(Label(text='UI Options', font_size=16, bold=True, size_hint_y=None, height=25))

        def create_switch(text, key):
            switch_layout = BoxLayout(orientation='horizontal', size_hint_y=None, height=30, spacing=10)
            switch_layout.add_widget(Label(text=text, size_hint_y=None, height=25))
            switch = Switch(active=self.shared_config[key], size_hint_y=None, height=30)
            switch.bind(active=lambda s, v: self.update_config(key, v))
            switch_layout.add_widget(switch)
            layout.add_widget(switch_layout)

        create_switch('Minimap', 'show_minimap')
        create_switch('HUD', 'show_hud')

        # Controls Info
        layout.add_widget(Label(text='Controls:', font_size=16, bold=True, size_hint_y=None, height=25))
        layout.add_widget(Label(
            text='WASD - Move\nMouse - Rotate\nLeft-Click - Shoot',
            font_size=14, size_hint_y=None, height=50, halign='center'
        ))

        scroll.add_widget(layout)
        return scroll
    
    
    def update_config_and_label(self, key, value, label):
        self.update_config(key, value)
        if key == 'FOV':
            label.text = f'{value:.1f}'
        elif key == 'num_enemies':
            label.text = str(int(value))
        elif isinstance(value, float):
            label.text = f'{value:.1f}'
        else:
            label.text = str(value)
    
    
    def update_config(self, key, value):
        if key == 'FOV':
            value = math.radians(value)
        self.shared_config[key] = value
    
    
    def start_game(self, instance):
        if self.game_proc_container[0] is None or not self.game_proc_container[0].is_alive():
            proc = mp.Process(target=game_process, 
                            args=(self.shared_config, self.command_queue))
            proc.start()
            self.game_proc_container[0] = proc
            self.status_label.text = 'Status: Running'
    
    def stop_game(self, instance):
        if self.game_proc_container[0] and self.game_proc_container[0].is_alive():
            self.command_queue.put('STOP')
            self.game_proc_container[0].join(timeout=2)
            if self.game_proc_container[0].is_alive():
                self.game_proc_container[0].terminate()
            self.game_proc_container[0] = None
            self.status_label.text = 'Status: Stopped'
    
    def restart_game(self, instance):
        if self.game_proc_container[0] and self.game_proc_container[0].is_alive():
            self.command_queue.put('RESTART')


class DoomLauncherApp(App):
    def __init__(self, shared_config, command_queue, game_proc_container, **kwargs):
        super().__init__(**kwargs)
        self.shared_config = shared_config
        self.command_queue = command_queue
        self.game_proc_container = game_proc_container
    
    def build(self):
        # Load KV file
        Builder.load_file('launcher.kv')
        return LauncherUI(self.shared_config, self.command_queue, self.game_proc_container)
    
    def on_stop(self):
        # Clean up game process on exit
        if self.game_proc_container[0] and self.game_proc_container[0].is_alive():
            self.command_queue.put('STOP')
            self.game_proc_container[0].join(timeout=2)
            if self.game_proc_container[0].is_alive():
                self.game_proc_container[0].terminate()


# MAIN ENTRY POINT

def main():
    # Setup multiprocessing
    mp.set_start_method('spawn', force=True)
    
    # Shared configuration
    manager = mp.Manager()
    shared_config = manager.dict(DEFAULT_CONFIG)
    command_queue = manager.Queue()
    
    # Container for game process (mutable reference)
    game_proc_container = [None]
    
    # Start Kivy UI
    app = DoomLauncherApp(shared_config, command_queue, game_proc_container)
    app.run()


if __name__ == '__main__':
    main()
