"""
Doom Game Logic - With mouse control and FOV settings
"""

import pygame
import math
import random
import time
from queue import Empty


# PLAYER CLASS - With Mouse Rotation

class Player:
    def __init__(self, x, y, angle, health=100):
        self.x = x
        self.y = y
        self.angle = angle
        self.health = health
        
    def move(self, keys, dt, speed, rotation_speed, game_map, config):
        move_speed = speed * dt
        dx = math.cos(self.angle) * move_speed
        dy = math.sin(self.angle) * move_speed
        
        # Movement with WASD
        if keys[config['key_forward']]:
            new_x, new_y = self.x + dx, self.y + dy
            if not self.check_collision(new_x, new_y, game_map):
                self.x, self.y = new_x, new_y
                
        if keys[config['key_backward']]:
            new_x, new_y = self.x - dx, self.y - dy
            if not self.check_collision(new_x, new_y, game_map):
                self.x, self.y = new_x, new_y
        
        # Strafing (A/D keys)
        if keys[config['key_right']]:
            new_x, new_y = self.x - dy, self.y + dx
            if not self.check_collision(new_x, new_y, game_map):
                self.x, self.y = new_x, new_y
                
        if keys[config['key_left']]:
            new_x, new_y = self.x + dy, self.y - dx
            if not self.check_collision(new_x, new_y, game_map):
                self.x, self.y = new_x, new_y
    
    def mouse_rotate(self, rel_x, sensitivity=0.005):
        """Mouse-based rotation with relative movement"""
        rotation_amount = rel_x * sensitivity
        self.angle += rotation_amount

        # Angle normalization
        while self.angle > math.pi:
            self.angle -= 2 * math.pi
        while self.angle < -math.pi:
            self.angle += 2 * math.pi
    
    def check_collision(self, x, y, game_map):
        map_x, map_y = int(x), int(y)
        map_size = len(game_map)
        if 0 <= map_x < map_size and 0 <= map_y < map_size:
            return game_map[map_y][map_x] == 1
        return True


# ENEMY CLASS

class Enemy:
    def __init__(self, x, y):
        self.x = x
        self.y = y
        self.alive = True

    def move(self, dt, player, speed, game_map):
        if not self.alive:
            return
        dx = player.x - self.x
        dy = player.y - self.y
        distance = math.sqrt(dx**2 + dy**2)
        if distance > 0:
            move_speed = speed * dt
            dir_x = dx / distance
            dir_y = dy / distance
            new_x = self.x + dir_x * move_speed
            new_y = self.y + dir_y * move_speed
            if not self.check_collision(new_x, new_y, game_map):
                self.x = new_x
                self.y = new_y

    def check_collision(self, x, y, game_map):
        map_x, map_y = int(x), int(y)
        map_size = len(game_map)
        if 0 <= map_x < map_size and 0 <= map_y < map_size:
            return game_map[map_y][map_x] == 1
        return True

    def draw(self, screen, player, width, height, fov, game_map):
        if not self.alive:
            return

        dx = self.x - player.x
        dy = self.y - player.y
        distance = math.sqrt(dx**2 + dy**2)

        if distance < 0.1:
            return

        angle = math.atan2(dy, dx) - player.angle

        while angle > math.pi:
            angle -= 2 * math.pi
        while angle < -math.pi:
            angle += 2 * math.pi

        half_fov = fov / 2
        if abs(angle) < half_fov + 0.5:
            # Check visibility through walls
            if not self.is_visible_behind_walls(player, game_map):
                return

            screen_x = (angle / fov + 0.5) * width
            sprite_height = min(height, height / (distance + 0.0001)) / 2  # Halb so groß
            sprite_width = sprite_height * 0.7  # Schmaler

            enemy_rect = pygame.Rect(
                screen_x - sprite_width // 2,
                height // 2 - sprite_height // 2,
                sprite_width,
                sprite_height
            )
            pygame.draw.rect(screen, (0, 0, 255), enemy_rect)  # Blau

    def is_visible_behind_walls(self, player, game_map):
        dx = self.x - player.x
        dy = self.y - player.y
        dist_to_enemy = math.sqrt(dx**2 + dy**2)
        angle = math.atan2(dy, dx)

        max_depth = 20  # Adjust as needed
        map_size = len(game_map)

        for depth in range(1, int(max_depth * 10)):
            depth_val = depth / 10
            target_x = player.x + depth_val * math.cos(angle)
            target_y = player.y + depth_val * math.sin(angle)

            map_x, map_y = int(target_x), int(target_y)

            if 0 <= map_x < map_size and 0 <= map_y < map_size:
                if game_map[map_y][map_x] == 1:
                    if depth_val < dist_to_enemy:
                        return False  # Wall blocks

        return True


# RAYCASTING ENGINE - With FOV Support

def cast_rays(player, game_map, config):
    """Raycasting algorithm with dynamic FOV"""
    walls = []
    fov = config['FOV']
    num_rays = int(config['NUM_RAYS'])
    max_depth = config['MAX_DEPTH']
    width = int(config['WIDTH'])
    height = int(config['HEIGHT'])
    
    half_fov = fov / 2
    delta_angle = fov / num_rays
    scale = width // num_rays
    
    ray_angle = player.angle - half_fov
    map_size = len(game_map)
    
    for ray in range(num_rays):
        # Optimized ray calculation
        for depth in range(1, int(max_depth * 10)):
            depth_val = depth / 10
            target_x = player.x + depth_val * math.cos(ray_angle)
            target_y = player.y + depth_val * math.sin(ray_angle)
            
            map_x, map_y = int(target_x), int(target_y)
            
            if 0 <= map_x < map_size and 0 <= map_y < map_size:
                if game_map[map_y][map_x] == 1:
                    # Fish-eye correction
                    depth_val *= math.cos(player.angle - ray_angle)
                    wall_height = min(height, height / (depth_val + 0.0001))

                    # Simple stone wall color
                    color = (178, 34, 52) #Color for walls

                    walls.append((ray * scale, wall_height, color, depth_val))
                    break
        
        ray_angle += delta_angle
    
    return walls


# RENDERING FUNCTIONS

def draw_3d_view(screen, walls, config):
    """Draws the 3D view"""
    width = int(config['WIDTH'])
    height = int(config['HEIGHT'])

    # Simple sky and ground
    pygame.draw.rect(screen, (135, 206, 235), (0, 0, width, height // 2))  # Sky blue
    pygame.draw.rect(screen, (100, 100, 100), (0, height // 2, width, height // 2))  # Ground gray

    # Draw walls
    scale = width // int(config['NUM_RAYS'])
    for wall_x, wall_height, color, depth in walls:
        pygame.draw.rect(screen, color,
                        (wall_x, height // 2 - wall_height // 2,
                         scale + 1, wall_height))


def draw_minimap(screen, player, enemies, game_map, config):
    """Draws minimap"""
    if not config['show_minimap']:
        return
    
    minimap_size = 120
    map_size = len(game_map)
    tile_size = minimap_size // map_size
    offset_x, offset_y = 10, 10
    
    # Hintergrund mit Rahmen
    pygame.draw.rect(screen, (255, 255, 255), 
                    (offset_x-2, offset_y-2, minimap_size+4, minimap_size+4))
    pygame.draw.rect(screen, (0, 0, 0), 
                    (offset_x, offset_y, minimap_size, minimap_size))
    
    # Map Tiles
    for y in range(map_size):
        for x in range(map_size):
            if game_map[y][x] == 1:
                # Walls as white blocks
                color = (255, 255, 255)
            else:
                # Empty space as dark dots
                color = (100, 100, 100)
            
            pygame.draw.rect(screen, color,
                           (offset_x + x * tile_size,
                            offset_y + y * tile_size,
                            tile_size - 1, tile_size - 1))
    
    # Player as green circle
    player_x = offset_x + int(player.x * tile_size)
    player_y = offset_y + int(player.y * tile_size)
    pygame.draw.circle(screen, (0, 255, 0), (player_x, player_y), 3)

    # Player direction
    direction_x = player_x + math.cos(player.angle) * 8
    direction_y = player_y + math.sin(player.angle) * 8
    pygame.draw.line(screen, (0, 255, 0), (player_x, player_y), 
                    (direction_x, direction_y), 2)


def draw_hud(screen, player, font, config):
    """Draws simple HUD"""
    if not config['show_hud']:
        return

    width = int(config['WIDTH'])
    height = int(config['HEIGHT'])

    # Health Text
    health_text = font.render(f"Health: {int(player.health)}", True, (255, 255, 255))
    screen.blit(health_text, (width - 100, height - 30))

    # Simple health bar
    bar_x, bar_y, bar_width, bar_height = width - 100, height - 20, 80, 10
    pygame.draw.rect(screen, (255, 0, 0), (bar_x, bar_y, bar_width, bar_height))  # Background red
    pygame.draw.rect(screen, (0, 255, 0), (bar_x, bar_y, int(player.health * 0.8), bar_height))  # Health green

    # Simple crosshair
    center_x, center_y = width // 2, height // 2
    pygame.draw.line(screen, (255, 255, 255), (center_x - 8, center_y), (center_x + 8, center_y), 2)
    pygame.draw.line(screen, (255, 255, 255), (center_x, center_y - 8), (center_x, center_y + 8), 2)


# UTILITY FUNCTIONS

def spawn_enemies(config):
    """Spawns enemies at random positions"""
    game_map = config['map']
    map_size = len(game_map)
    num_enemies = int(config['num_enemies'])
    enemies = []
    
    for _ in range(num_enemies):
        attempts = 0
        while attempts < 50:
            x = random.randint(1, map_size - 2)
            y = random.randint(1, map_size - 2)
            if game_map[y][x] == 0:
                enemies.append(Enemy(x + 0.5, y + 0.5))
                break
            attempts += 1
    
    return enemies


def validate_spawn(x, y, game_map):
    """Validates spawn position"""
    map_x, map_y = int(x), int(y)
    map_size = len(game_map)
    if 0 <= map_x < map_size and 0 <= map_y < map_size:
        return game_map[map_y][map_x] == 0
    return False


# MAIN GAME PROCESS - With Mouse Control

def game_process(shared_config, command_queue):
    """Main game loop - with mouse control"""
    pygame.init()

    # Load and type conversion
    config = {k: v for k, v in shared_config.items()}
    config['WIDTH'] = int(config['WIDTH'])
    config['HEIGHT'] = int(config['HEIGHT'])
    config['NUM_RAYS'] = int(config['NUM_RAYS'])

    # Increase default rays for better quality
    if config['NUM_RAYS'] < 200:
        config['NUM_RAYS'] = 200
    
    # Pygame Setup
    screen = pygame.display.set_mode((config['WIDTH'], config['HEIGHT']))
    pygame.display.set_caption("Doom")
    clock = pygame.time.Clock()
    font = pygame.font.Font(None, 24)

    # Enable mouse support
    pygame.mouse.set_visible(False)
    pygame.event.set_grab(True)

    # Validate spawn position
    spawn_x, spawn_y = config['player_start_x'], config['player_start_y']
    if not validate_spawn(spawn_x, spawn_y, config['map']):
        # Find first valid position
        for y in range(len(config['map'])):
            for x in range(len(config['map'][0])):
                if config['map'][y][x] == 0:
                    spawn_x, spawn_y = x + 0.5, y + 0.5
                    break

    # Initialize player and enemies
    player = Player(spawn_x, spawn_y, config['player_start_angle'], config['player_health'])
    enemies = spawn_enemies(config)

    last_config_check = time.time()
    
    # Main Game Loop
    running = True
    while running:
        dt = clock.tick(60) / 1000.0
        
        # Command Handling
        try:
            cmd = command_queue.get_nowait()
            if cmd == 'STOP':
                running = False
            elif cmd == 'RESTART':
                player = Player(spawn_x, spawn_y, config['player_start_angle'], config['player_health'])
                enemies = spawn_enemies(config)
        except Empty:
            pass

        # Config update (simplified)
        if time.time() - last_config_check > 0.2:
            for key, value in shared_config.items():
                if key in ['WIDTH', 'HEIGHT', 'NUM_RAYS', 'num_enemies', 'FOV']:
                    if key == 'FOV':
                        config[key] = float(value)
                    else:
                        config[key] = int(value)
                else:
                        config[key] = value

            last_config_check = time.time()
        
        # Event Handling
        for event in pygame.event.get():
            if event.type == pygame.QUIT:
                running = False
            if event.type == pygame.MOUSEBUTTONDOWN and event.button == 1:  # Left-Click Shoot
                # Improved shoot logic with direction cone
                shoot_cone = math.radians(15)  # 15 degree cone
                shoot_range = 15  # max distance

                for enemy in enemies:
                    if enemy.alive:
                        dx = enemy.x - player.x
                        dy = enemy.y - player.y
                        dist = math.sqrt(dx**2 + dy**2)
                        if dist > 0 and dist <= shoot_range:
                            enemy_angle = math.atan2(dy, dx) - player.angle
                            while enemy_angle > math.pi:
                                enemy_angle -= 2 * math.pi
                            while enemy_angle < -math.pi:
                                enemy_angle += 2 * math.pi
                            if abs(enemy_angle) <= shoot_cone / 2:
                                enemy.alive = False
                                break  # Kill only the first hit enemy
        
        # Player Movement
        keys = pygame.key.get_pressed()
        player.move(keys, dt, config['player_speed'], 
                   config['player_rotation_speed'], config['map'], config)

        # Mouse rotation with correction for capture loss
        if not pygame.mouse.get_focused():
            # Mouse was released, recapture
            pygame.mouse.set_visible(False)
            pygame.event.set_grab(True)
            pygame.mouse.set_pos(config['WIDTH'] // 2, config['HEIGHT'] // 2)

        rel_x, rel_y = pygame.mouse.get_rel()
        player.mouse_rotate(rel_x)

        # Set mouse back to center for unlimited movement
        pygame.mouse.set_pos(config['WIDTH'] // 2, config['HEIGHT'] // 2)

        # Enemy Movement
        for enemy in enemies:
            enemy.move(dt, player, config['player_speed'] / 2, config['map'])

        # RENDERING
        screen.fill((0, 0, 0))

        # 1. Raycasting
        walls = cast_rays(player, config['map'], config)
        draw_3d_view(screen, walls, config)

        # 2. Enemies
        visible_enemies = [e for e in enemies if e.alive]
        visible_enemies.sort(key=lambda e: math.sqrt((e.x-player.x)**2 + (e.y-player.y)**2), reverse=True)
        for enemy in visible_enemies:
            enemy.draw(screen, player, config['WIDTH'], config['HEIGHT'], config['FOV'], config['map'])
        
        # 3. UI
        draw_minimap(screen, player, enemies, config['map'], config)
        draw_hud(screen, player, font, config)
        
        
        pygame.display.flip()
    
    pygame.mouse.set_grab(False)
    pygame.quit()
